<?php
namespace Saltus\WP\Plugin\Saltus\InteractiveGlobes\Features\LiveFilter;

if ( defined( 'SALTUS_PLAN' ) && SALTUS_PLAN !== 'pro' ) {
	return [];
}

use Saltus\WP\Framework\Infrastructure\Plugin\Registerable;
use Saltus\WP\Plugin\Saltus\InteractiveGlobes\Services\Assets\AssetsCore;
use Saltus\WP\Plugin\Saltus\InteractiveGlobes\Project;
use Saltus\WP\Framework\Infrastructure\Container\Invalid;
use Saltus\WP\Framework\Infrastructure\Container\ServiceContainer;
/**
 * The Meta class
 */
class LiveFilterCore implements Registerable {

	private $services;
	private $project;
	private $meta;
	private $options;

	public function __construct( ...$dependencies ) {

		if ( empty( $dependencies[0] ) ) {
			throw Invalid::from( 'Project' );
		}
		if ( ! $dependencies[0] instanceof Project ) {
			throw Invalid::from( $dependencies[0] );
		}

		if ( empty( $dependencies[1] ) ) {
			throw Invalid::from( 'ShortcodeCore' );
		}
		if ( ! $dependencies[1] instanceof ServiceContainer ) {
			throw Invalid::from( $dependencies[1] );
		}

		$this->project  = $dependencies[0];
		$this->services = $dependencies[1];

	}

	/**
	 * Register Tooltip filters
	 */
	public function register() {
		// single globe page filter
		add_action('itt_globes/render/post_setup_meta', [$this, 'check_filter'], 50, 2);

		$this->options = get_option( 'ittglobes' );
	}

	public function check_filter( $meta, $id ) {
		if ( empty( $meta['liveFilter'] ) || boolval( $meta['liveFilter']['enabled'] ) === false ) {
			return $meta;
		}

		$hook = 'itt_globes/render/content_before';

		if( $meta['liveFilter']['position'] === 'below' ){
			
			$hook = 'itt_globes/render/content_after';
		}

		$meta['id'] = $id;

		$this->meta = $meta;

		add_filter( 'itt_globes/render/container_class', [ $this, 'container_class' ] );
		add_action( $hook, [$this, 'filter_html_content'], 10, 2 );
		add_action( $hook, [$this, 'filter_styles'], 20, 2 );

		$this->register_assets( );

		return $meta;
	}

	public function filter_styles( $content, $id ) {

		// menuItemColor and menuItemTextColor
		$filter_menu_styles = $this->options['filterMenuStyles'] ?? [];

		$menu_item_color = $filter_menu_styles['menuItemColor']['color'] ?? '#f5f5f5';
		$menu_item_text_color = $filter_menu_styles['menuItemTextColor']['color'] ?? '#000000';

		$menu_item_color_hover = $filter_menu_styles['menuItemColor']['hover'] ?? '#cccccc';
		$menu_item_text_color_hover = $filter_menu_styles['menuItemTextColor']['hover'] ?? '#000000';

		$menu_item_color_active = $filter_menu_styles['menuItemColor']['active'] ?? '#cccccc';
		$menu_item_text_color_active = $filter_menu_styles['menuItemTextColor']['active'] ?? '#000000'	;

		$menu_item_border_radius = $filter_menu_styles['borderRadius'] ?? 3;

		$styles = sprintf(
			'<style>
				.itt-globe-live-filter .itt_globe_filter_item {
					background-color: %1$s;
					color: %2$s;
					border-radius: %3$spx;
				}

				.itt-globe-live-filter .itt_globe_filter_item:hover {
					background-color: %4$s;
					color: %5$s;
				}

				.itt-globe-live-filter .itt_globe_filter_item_active {
					background-color: %6$s;
					color: %7$s;
				}
			</style>',
			$menu_item_color,
			$menu_item_text_color,
			$menu_item_border_radius,
			$menu_item_color_hover,
			$menu_item_text_color_hover,
			$menu_item_color_active,
			$menu_item_text_color_active
		);

		if ( $content !== $styles ) {
			$content = $styles . $content;
		}
		return $content;
	}

	public function register_assets( ) {
		try {
			$assets = $this->services->get( 'assets' )->create();
		} catch ( \Exception $exception ) {

			if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions
				error_log( 'Failed to load styles' );
			}
		}

		$name = $assets->register_script(
			'/assets/js/public/features/filter/filter.js',
			[],
			true
		);
		wp_enqueue_script( $name );

		$name = $assets->register_style(
			'/assets/css/features/filter/filter.css',
			[]
		);
		wp_enqueue_style( $name );

	}

	public function container_class( $container_class ) {
		return $container_class . ' itt_globe_wrapper_with_filter';
	}

	public function filter_html_content( $content, $id ) {
		$meta = $this->meta;

		// right now we only support filter by layers, so we check the layers
		if( empty( $meta['layer'] ) ){
			return $content;
		}

		$layers_html = '';

		$filter_type = isset($meta['liveFilter']['type']) ? $meta['liveFilter']['type'] : 'menu';

		switch ($filter_type) {
			case 'menu':
				$layers_html .= '<div class="itt_globe_filter_menu_container">' . $this->create_menu( $meta, $id ) . '</div>';
				break;

			case 'dropdown':
				$layers_html .= '<div class="itt_globe_filter_dropdown_container">' . $this->create_dropdown( $meta, $id ) . '</div>';
				break;

			case 'menu_dropdown':
				$layers_html .= '<div class="itt_globe_filter_menu_dropdown_container">' . $this->create_menu( $meta, $id ) . $this->create_dropdown( $meta, $id ) . '</div>';
				break;
			
			case 'toggle':
				$layers_html .= '<div class="itt_globe_filter_toggle_container">' . $this->create_menu( $meta, $id, 'toggle' ) . '</div>';
				break;

			default:
				// If an unsupported type is specified, default to menu
				
				break;
		}

		$filter_html = '<div class="itt-globe-live-filter">' . $layers_html . '</div>';
		
		return $filter_html . $content;
	}

	public function create_dropdown($meta, $id) {
		$layers = $meta['layer'];
		$options_html = '';

		// All option
		if (!empty($meta['liveFilter']['allLabel'])) {
			$selected = intval($meta['liveFilter']['default']) === intval($id) ? 'selected' : '';
			$options_html .= sprintf(
				'<option value="%2$s" %3$s>%1$s</option>',
				$meta['liveFilter']['allLabel'],
				$id,
				$selected
			);
		}

		foreach ($layers as $layer) {
			$globe_name = get_the_title($layer);
			$selected = intval($meta['liveFilter']['default']) === intval($layer) ? 'selected' : '';

			$options_html .= sprintf(
				'<option value="%2$s" %3$s>%1$s</option>',
				$globe_name,
				$layer,
				$selected
			);
		}

		$keep_base = $meta['liveFilter']['keepBase'] == "1" ? $id : '0';
		$filter_html = sprintf(
			'<select data-globe-id="%1$s" data-keep-base="%3$s" class="itt_globe_filter_dropdown">%2$s</select>',
			$id,
			$options_html,
			$keep_base
		);

		return $filter_html;
	}

	public function create_menu( $meta, $id, $type = 'filter' ){

		$layers = $meta['layer'];
		$layers_html = '';

		// all option
		if( ! empty ( $meta['liveFilter']['allLabel'] ) && $type !== 'toggle' ) {

			$class     = intval( $meta['liveFilter']['default'] ) === intval( $id ) ? 'itt_globe_filter_item_active' : '';
			
			$layers_html .= sprintf(
				'<div data-globe-id="%2$s" class="itt_globe_filter_item %3$s"> %1$s </div>',
				$meta['liveFilter']['allLabel'],
				$id,
				$class
			);
		}

		foreach( $layers as $layer ){
			$globe_name = get_the_title( $layer );
			$class = intval( $meta['liveFilter']['default'] ) === intval( $layer ) ? 'itt_globe_filter_item_active' : '';
			

			$layers_html .= sprintf(
				'<div data-globe-id="%2$s"  class="itt_globe_filter_item %3$s"> %1$s </div>',
				$globe_name,
				$layer,
				$class
			);
		}

		$type_class = $type === 'filter' ? 'itt_globe_filter_menu' : 'itt_globe_' . $type . '_menu';

		$keep_base = $meta['liveFilter']['keepBase'] == "1" ? $id : '0';
		$filter_html = sprintf( '<div data-globe-id="%1$s" data-keep-base="%3$s" class="%4$s" >%2$s </div>', $id, $layers_html, $keep_base, $type_class);
		return $filter_html;
	}
}
