/** Setup tooltip for element */
function setupTooltip( element, template ) {

	let tooltip = element.tooltipContent ?? '';
	if( typeof template !== 'undefined' && template !== '{tooltipContent}' && template.length > 0 ){

		tooltip = template.replace(/\{([^}]+)\}/g, function(match, key) {
			// Split the key by dot notation
			const keys = key.split('.');
			let value = element;

			// Traverse the object using the keys
			for (let i = 0; i < keys.length; i++) {
				value = value ? value[keys[i]] : undefined;
			}

			// If the value is undefined, replace it with an empty string or handle as necessary
			return value !== undefined ? value : '';
		});
	}

	if ( tooltip !== '') {
		tooltip = '<div class="itt_globe_' + element.globe_id + ' itt_globe_tooltip">' + tooltip + '</div>';
	}
	return tooltip;
}

/**
 * Prepares safe url
 * @param string str url to prepare
 * @returns string prepared URL
 */
const globe_ut_prepareURL = function (str) {

	if (typeof str !== "string") {
		return str;
	}

	// parse html entities
	str = str.replace(/&amp;/gi, '&');
	str.replace(/&#(\d+);/g, function (match, dec) {
		return String.fromCharCode(dec);
	});

	// check if allowed url
	var url, protocols;
	try {
		url = new URL(str);
	} catch (_) {
		url = false;
	}

	// check if valid url. If string it might be a anchor or relative path, so keep going
	if (url) {
		// acepted protocols
		protocols = [null, "http:", "https:", "mailto:", "tel:"];
		if (!protocols.includes(url.protocol)) {
			console.log('URL protocol not allowed');
			return '';
		}
	}

	return str;

};

/** Prepare data to be used. Set defaults and other operations */
const globe_ut_prepareData = function ( data ) {

	if ( ! data.hasOwnProperty( 'dotLabels' ) || ! data.dotLabels ) {
		data['dotlabels'] = [];
	}
	if ( ! data.hasOwnProperty( 'points' ) || ! data.points ) {
		data['points'] = [];
	}

	// set preview defaults
	let defaults = {
		'showGraticules': false,
		'backgroundColor': 'transparent',
		'atmosphere': {
			'enabled': false,
			'atmosphereColor': null,
			'atmosphereAltitude': null
		},
		'globeImage': 'earth-day.jpg'
	};

	// merge data with defaults
	if ( Object.keys(data).length === 0 || data.emptyPreview ) {
		Object.assign(data, defaults);
	}



	// set point of view
	data.pointOfView = {
		lat: 0,
		lng: 0,
		altitude: 1.5
	};

	// shouldn't be empty
	if (data.centerCoordinates) {
		data.pointOfView.lat      = globe_ut_float(data.centerCoordinates.latitude);
		data.pointOfView.lng      = globe_ut_float(data.centerCoordinates.longitude);
		data.pointOfView.altitude = globe_ut_float(data.altitude);
	}

	return data;
};

const globe_ut_prepareProData = function ( data ) {

	if ( ! data.hasOwnProperty( 'arcLines' ) || ! data.arcLines ) {
		data['arcLines'] = [];
	}
	if ( ! data.hasOwnProperty( 'regions' ) || ! data.arcLines ) {
		data['regions'] = [];
	}
	return data;
};


const globe_ut_setupData = function ( data, dataTypes ) {
	for (const key in dataTypes) {
		let defaults = dataTypes[key];
		if (Array.isArray(data[key]) && data[key]) {
			data[key].map(function (entry) {
				// add defaults data type
				if ( typeof entry.useCustom === "undefined" ||
					( typeof entry.useCustom !== "undefined" && ! globe_ut_bool( entry.useCustom ) ) ) {
					Object.assign(entry, data[defaults]);
				}
				// if it's using default action
				if (entry.action && entry.action === 'default') {
					entry.action = data[defaults]['action'];
				}
			});
		}
	}
};

/**
 * Prepare viewport and globe
 * @param DOM element container
 * @param object data
 * @param Globe Object thisGlobe
 */
const globe_ut_prepareViewport = function ( container, data ) {
	var aspRatioContainer = container.closest(".itt_globe_aspect_ratio");

	// container size adjustment
	// if mobile
	if (window.innerWidth <= 780 && typeof data.paddingTop !== 'undefined' && data.paddingTop !== '') {
		aspRatioContainer.style.paddingTop = String(data.paddingTopMobile) + '%';
	} else {
		aspRatioContainer.style.paddingTop = String(data.paddingTop) + '%';
	}
};

/**
 * Returns float, even if not a number
 * @param {string} string to get float from
 * @returns int float number
 */
const globe_ut_float = function (string) {
	var float = parseFloat(string);
	if (isNaN(float)) {
		return 0;
	}
	return float;
};

/**
 * Returns boolean value of a string or number
 * @param {*} string
 * @returns boolen
 */
const globe_ut_bool = function (string) {
	var bool = Number(string) === 0 || string === "false" || typeof string === "undefined" ? false : true;
	return bool;
};

const globe_ut_prepare_coordinates = function( type, marker ){

	if( typeof marker.coordinates !== 'undefined' && marker.coordinates[type] ){
		return parseFloat( marker.coordinates[type] );
	}

	if( typeof marker[type] !== 'undefined' ){
		return parseFloat( marker[type] );
	}

	return false;
	
};

/** Setup click events for element */
function setupClickEvent( thisGlobe, GlobeData, element ) {

	let layerDataTypes = {
		'arcLines': 'arcLineDefaults',
		'points': 'pointDefaults',
		'dotLabels': 'labelDefaults',
		'regions': 'regionDefaults',
	};

	if(element.type === 'regions'){
		element = { ...element.properties };
	}

	if (new URLSearchParams(window.location.search).has('debug')) {
		console.debug(element);
	}

	// if we arrive here and it's still set as default, we need to set it to the default action for that layer
	if( element.action === 'default' ){
		element.action = GlobeData[layerDataTypes[element.type]].action;
	}

	if (element.action === "none") {
		// do nothing
		return;
	}

	const clickAction = element.action;
	if (clickAction && typeof window.ittGlobes.clickActions[clickAction] === 'function') {
		window.ittGlobes.clickActions[clickAction]( thisGlobe, GlobeData, element );
	}
	if ( ittGlobeData.isAdmin ) {
		return;
	}

	// check urls
	if (element.action === "open_url" || element.action === "open_url_new") {
		element.content = globe_ut_prepareURL(element.content);
	}
	// open new url
	if (element.action === "open_url" && element.content !== "") {
		document.location = element.content;
	} else if (element.action === "open_url_new" && element.content !== "") {
		window.open(element.content);
	}
}

/** Setup hover events for element */
function setupHoverEvent( thisGlobe, element, type ) {

	let GlobeData = thisGlobe.meta;
	if(type === 'regions'){
		thisGlobe.polygonCapColor(function(region){
			// when hovering the globe itself, element doesn't exist or doesn't have properties
			if( typeof element === 'undefined' || ( element && typeof element.properties === 'undefined' )){
				return region.properties.color ? region.properties.color : GlobeData.regionDefaults.inactive;
			}
			// change hover color if being hovered or part of same group
			else if( region === element){
				return region.properties.hover;
			} else if (region.properties.originalId && element && element.properties && element.properties.id && region.properties.originalId.includes( element.properties.id )){
				return region.properties.hover;
			}
			else if (region.properties.originalId && element && element.properties && Array.isArray(element.properties.id) && element.properties.id.includes(region.properties.id)){
				return region.properties.hover;
			}
			// we actually don't need the previous I think, we can reset the value for the rest of the entries
			return region.properties.color ? region.properties.color : GlobeData.regionDefaults.inactive
		});
	} else {

		thisGlobe.labelColor(function(el){
			return element && el.id === element.id ? el.hover : el.color;
		});

		thisGlobe.pointColor(function(el){
			return element && el.id === element.id ? el.hover : el.color;
		});

		//pause/resume animation
		if (typeof GlobeData.rotate !== 'undefined' && globe_ut_bool(GlobeData.rotate.enabled)) {
			thisGlobe.controls().autoRotate = !element;
		}
	}

	return thisGlobe;

}

/** Add Points Layer to Globe */
function addPointsLayer ( thisGlobe, GlobeData ) {

	thisGlobe.pointsData( GlobeData.points )
		.pointLat(p => globe_ut_prepare_coordinates('latitude', p) )
		.pointLng(p => globe_ut_prepare_coordinates('longitude', p) )
		.pointRadius(p => p.radius / 10)
		.pointAltitude(p => p.altitude / 100)
		.pointColor(p => p.color)
		.pointResolution(20);

	// tooltip
	thisGlobe.pointLabel(function (label) {

		let pointsTooltipTemplate = GlobeData.tooltipTemplate ?? '{tooltipContent}';
		if( GlobeData.pointsTooltipTemplate ){
			pointsTooltipTemplate = GlobeData.pointsTooltipTemplate;
		}
		return setupTooltip(label, pointsTooltipTemplate );

	});

	// click events
	thisGlobe.onPointClick(function ( element ) {
		setupClickEvent( thisGlobe, GlobeData, element );
	});

	// hover events
	thisGlobe.onPointHover(function (label) {
		setupHoverEvent(thisGlobe, label, 'point');
	});
}

/** Add Labels Layer to Globe */
function addLabelsLayer( thisGlobe, GlobeData ) {

	thisGlobe.labelsData( GlobeData.dotLabels )
		.labelLat(l => globe_ut_prepare_coordinates('latitude', l))
		.labelLng(l => globe_ut_prepare_coordinates('longitude', l))
		.labelText(l => l.title ?? '' )
		.labelAltitude(l => l.altitude / 100)
		.labelSize(l => l.size / 10)
		.labelColor(l => l.color)
		.labelResolution(20)
		.labelIncludeDot(l => globe_ut_bool(l.includeDot))
		.labelDotRadius(l => l.radius / 10)
		.labelDotOrientation(l => l.dotOrientation);

	// font
	if (GlobeData.labelFont && GlobeData.labelFont !== 'default') {

		let url = ittGlobeData.assetsUrl + 'fonts/' + GlobeData.labelFont + '.json';

		fetch(url)
			.then(res => res.json())
			.then( labelfont => { thisGlobe.labelTypeFace(labelfont); } )
			.catch(error => {
				console.error('Fetch error:', error);
			});
	}

	// tooltip
	thisGlobe.labelLabel(function (label) {

		let dotLabelsTooltipTemplate = GlobeData.tooltipTemplate ?? '{tooltipContent}';
		if( GlobeData.dotLabelsTooltipTemplate ){
			dotLabelsTooltipTemplate = GlobeData.dotLabelsTooltipTemplate;
		}
		return setupTooltip(label, dotLabelsTooltipTemplate );
	});

	// click events
	thisGlobe.onLabelClick(function (label, event, coordinates) {
		setupClickEvent( thisGlobe, GlobeData, label );
	});

	// hover events
	thisGlobe.onLabelHover(function (label) {
		setupHoverEvent(thisGlobe, label, 'label');
	});
}

/** Setup click events for element */
function setupResizeEvent( thisGlobe, container, data ) {

	var aspRatioContainer = container.closest(".itt_globe_aspect_ratio");

	window.addEventListener('resize', function () {
		if (window.innerWidth <= 780 && typeof data.paddingTop !== 'undefined' && data.paddingTop !== '') {
			aspRatioContainer.style.paddingTop = String(data.paddingTopMobile) + '%';
		} else {
			aspRatioContainer.style.paddingTop = String(data.paddingTop) + '%';
		}

		// Here should go the code to rebuild globe with new size in a more performant way
		// check for size changes, etc..
		if (thisGlobe.width() !== container.offsetWidth) {
			thisGlobe
				.width(container.offsetWidth)
				.height(container.offsetHeight);
		}
	});
}

/**
 * Controls globe interactions
 * @param {*} thisGlobe
 * @param {*} data
 */
function setupInteractions( thisGlobe, data ) {
		// interactions
		if( typeof data.interactions !== 'undefined' ) {
			thisGlobe.controls().enableZoom   = globe_ut_bool( data.interactions.zoom );
			thisGlobe.controls().enablePan    = globe_ut_bool( data.interactions.pan );
			thisGlobe.controls().enableRotate = globe_ut_bool( data.interactions.pan );
		}
}

let ittGlobes$1 = {
	'globes': [],
	'globesIndex': {},
	'clickActions': {},
};

/** initialize globe */
ittGlobes$1.init = function () {
	// return early if Globe.gl Library didn't load properly
	if (typeof Globe === 'undefined') {
		console.error('Globe lib not loaded properly');
		return;
	}

	// check for globe container created by the shortcode
	let globe_render_container = document.getElementsByClassName( 'js-itt-globe-render' );
	for (const globe_container of globe_render_container) {
		if ( ! globe_container.dataset.hasOwnProperty('globe_id') ) {
			continue;
		}
		let globe_id = 0;
		try {
			globe_id = parseInt( JSON.parse(globe_container.dataset.globe_id ) );
		} catch (e) {
			continue;
		}

		if ( ittGlobes$1.globesIndex[ globe_id ] !== undefined ) {
			continue;
		}

		if ( ! globe_container.dataset.hasOwnProperty( 'globe_meta' ) ) {
			continue;
		}

		let globe_meta = {};
		try {
			globe_meta = JSON.parse( globe_container.dataset.globe_meta );
		} catch (e) {
			globe_meta = {};
		}
		create_globe( globe_id, globe_meta );
	}
};


var create_globe = function ( id, data ) {

	//data types - normal
	let dataTypes = {
		'points': 'pointDefaults',
		'dotLabels': 'labelDefaults',
	};
	let container = document.getElementById( 'itt_globe_' + id );

	globe_ut_prepareData( data );
	globe_ut_setupData( data, dataTypes );
	globe_ut_prepareViewport( container, data );

	let configOptions = {
		rendererConfig: { antialias: true, alpha: true },
		waitForGlobeReady: true,
		animateIn: typeof data.animateIn !== 'undefined' ? globe_ut_bool( data.animateIn ) : true,
	};

	let thisGlobe = Globe( configOptions )
		(document.getElementById( 'itt_globe_' + id ));

	// to make it publically accessible
	thisGlobe.meta = data;

	// background colour
	if (data.backgroundColor === 'transparent') {
		data.backgroundColor = 'rgba(0,0,0,0)';
	}

	// globe visuals
	if ( data.globeImage !== null && data.globeImage !== '' && data.globeImage !== 'earth-hollow' && data.globeImage !== 'noImage' && data.globeImage !== 'customImage' ) {
		thisGlobe.globeImageUrl( ittGlobeData.imagesUrl + data.globeImage );
	}

	thisGlobe.backgroundColor(data.backgroundColor)
		.width(container.offsetWidth)
		.height(container.offsetHeight)
		.showGraticules( globe_ut_bool(data.showGraticules))
		.showAtmosphere( globe_ut_bool(data.atmosphere.enabled))
		.atmosphereColor(data.atmosphere.atmosphereColor)
		.atmosphereAltitude(data.atmosphere.atmosphereAltitude)
		.pointOfView(data.pointOfView);

	setupInteractions( thisGlobe, data );
	setupResizeEvent( thisGlobe, container, data );

	// if admin
	if ( ittGlobeData.isAdmin ) {

		thisGlobe.onGlobeReady(function () {
			const event = new Event("adminGlobeReady");
			document.dispatchEvent(event);
		});
	}
	// setup points
	if ( data.points ) {
		addPointsLayer(thisGlobe, data);
	}

	// setup label points
	if ( data.dotLabels ) {
		addLabelsLayer( thisGlobe, data );
	}

	// add click event function to be accesible globally
	thisGlobe.setupClickEvent = setupClickEvent;
	thisGlobe.setupHoverEvent = setupHoverEvent;

	// add object to be public
	ittGlobes$1.globes.push( thisGlobe );
	ittGlobes$1.globesIndex[ id ] = thisGlobe;
};

ittGlobes$1.init();

window.ittGlobes = ittGlobes$1;

/** Add Arcs Layer to Globe - Pro */
function addArcsLayer( thisGlobe, data ) {

	const segmented_lines = processLinesArray(data.arcLines);

	thisGlobe.arcsData( segmented_lines )
		.arcStartLat(p => p.line[0].coordinates.latitude)
		.arcStartLng(p => p.line[0].coordinates.longitude)
		.arcEndLat(p => p.line[1].coordinates.latitude)
		.arcEndLng(p => p.line[1].coordinates.longitude)
		.arcColor( p => p.stroke )
		.arcStroke( p => p.strokeWidth )
		.arcDashLength( p => p.dashLength / 100 )
		.arcDashGap( p => p.dashGap / 100 )
		.arcDashAnimateTime( p => p.animate * 1000 )
		.arcAltitude(function (p) {
			if (isNaN(p.altitude) || p.altitude.trim() === '') {
				return null;
			}
			return p.altitude / 100;
		});
}

function processLinesArray(lines) {
	const segmented_lines = [];

	lines.forEach(entry => {
		const line_points = entry.line;

		if ( line_points.length <= 2) {
			segmented_lines.push(entry);
			return;
		}
		// create copy for all properties
		const arc_data = { ...entry };

		// replace the coordinates with a single line
		for (let i = 1; i < line_points.length; i++) {
			const new_arc_data = { ...arc_data };
			new_arc_data.line = [line_points[i - 1], line_points[i]];
			segmented_lines.push( new_arc_data );
		}
	});

	return segmented_lines;
}

/**
 * Adds geojson regions layer
 * @param {*} thisGlobe
 * @param {*} GlobeData
 */
const addRegionLayer = function( thisGlobe, GlobeData ) {

	// in case it's a hollow globe, we don't want to see the side color
	let sideColor = 'rgba(100, 100, 100, 0.05)';
	if( GlobeData.globeImage === 'earth-hollow'){
		sideColor = 'rgba(100, 100, 100, 0)';
	}

	return new Promise((resolve) => {
		thisGlobe.polygonAltitude( r => r.properties.altitude ?? 0.005 )
			.lineHoverPrecision(0)
			.polygonCapColor(r => r.properties.color ? r.properties.color : GlobeData.regionDefaults.inactive)
			.polygonSideColor(() => sideColor)
			.polygonStrokeColor(() => '#f0f0f0')
			.polygonCapCurvatureResolution(5)
			.polygonsTransitionDuration(300);

		thisGlobe.regionLayers = {};
		thisGlobe.originalPolygonsData = [];

		// tooltip
		thisGlobe.polygonLabel(function (region) {

			// if region was not added manually and option to only display on active region is enabled, skip
			let onlyActive = globe_ut_bool(GlobeData.tooltipOnlyActive);
			if( onlyActive && typeof region.properties.originalId === 'undefined' ){
				return;
			}
			// set tooltip template
			let regionsTooltipTemplate = GlobeData.tooltipTemplate ?? '{tooltipContent}';
			if( GlobeData.regionsTooltipTemplate ){
				regionsTooltipTemplate = GlobeData.regionsTooltipTemplate;
			}

			return setupTooltip(region.properties, regionsTooltipTemplate);
		});

		// click events
		thisGlobe.onPolygonClick(function (region) {
			setupClickEvent(thisGlobe, GlobeData, region.properties);
		});

		// hover events
		thisGlobe.onPolygonHover(function (current, previous) {
			setupHoverEvent( thisGlobe, current, 'regions' );
		});

		if (!GlobeData.regions || GlobeData.regions.length === 0) {
			resolve();
			//return;
			//We can't return here, otherwise the empty regions will not load. If we add an option to hide emtpy regions, it would be here
		}

		// fetch regions
		(async () => {
			const regions = await prepareRegions(thisGlobe, GlobeData);
			thisGlobe.polygonsData(regions);
			thisGlobe.originalPolygonsData = regions;

			resolve();
		})();
	});
};

const prepareRegions = async function( thisGlobe, GlobeData, altitudeMultiplier ){

	// if it already exists, return it, otherwise fetch it
	if (thisGlobe.regionLayers && thisGlobe.regionLayers[GlobeData.id]) {
		return thisGlobe.regionLayers[GlobeData.id];
	}

	altitudeMultiplier = altitudeMultiplier ?? 0;

	let geojson_source = GlobeData.regionSource;
	if ( geojson_source === 'disabled' || geojson_source === undefined ){
		return;
	}
	if ( GlobeData.regions === undefined ){
		return;
	}
	
	let fullpath;
	if( geojson_source !== 'custom' ){
		if ( ! geojson_source.includes('.geojson') && ! geojson_source.includes('.json') ) {
			geojson_source += '.json';
		}
		fullpath = ittGlobeData.assetsUrl + 'geojson/' + geojson_source;
	} else {
		fullpath = GlobeData.regionCustomSourceURL;
	}

	const selectedRegionsIndividually = GlobeData.regions.reduce((acc, region) => {

		// save original id, to use for grouped hover
		region.originalId = region.id;

		// Destructure the region to get all properties except ids
		const { id, ...regionProps } = region;
		if ( ! id ) {
			return acc;
		}
		// Create new objects with all region properties and the individual id
		const idsWithProps = id.map(id => ({ ...regionProps, id }));
		return acc.concat(idsWithProps);
	}, []);

	// Convert the selection of regions array to a dictionary for quick lookup
	const regionsDict = selectedRegionsIndividually.reduce((acc, item) => {
		// Add both id and name as keys if available
		acc[item.id] = item; // id from the regions object
		if (item.title) {     // title or name for country
			acc[item.title] = item;  // Store the title (e.g., "Algeria")
		}
		return acc;
	}, {});

	

	return fetch(fullpath)
		.then(res => res.json())
		.then(regionsAvailable => {
			// Enrich the GeoJSON data with each regions data
			regionsAvailable.features.forEach(feature => {
				const regionId = feature.id;
				const regionName = feature.properties.name;

				// Try to find the matching region data by id or name
				const regionData = regionsDict[regionId] || regionsDict[regionName];

				if (regionData) {
					feature.properties = {
						...feature.properties,
						...regionData
					};
				}
				// Add globe_id to feature properties
				feature.properties.globe_id = GlobeData.id;

				if (thisGlobe.liveFilter && thisGlobe.liveFilter.allLabel && thisGlobe.liveFilter.allLabel.trim() !== '') {
					feature.properties.altitude = 0.005 + ( 0.0025 * altitudeMultiplier );
				}
				
			});

			thisGlobe.regionLayers[GlobeData.id] = regionsAvailable.features;
			return regionsAvailable.features;
		});
};

/**
 * Adds Auto Rotate and its speed controls
 * @param {*} thisGlobe
 * @param {*} data
 */
function addAutoRotate( thisGlobe, data ) {
	if (typeof data.rotate !== 'undefined' && globe_ut_bool(data.rotate.enabled)) {
		thisGlobe.controls().autoRotate = true;
		thisGlobe.controls().autoRotateSpeed = data.rotate.speed;
	}
}

/**
 * Handles click actions to display content
 */
function displayContent( thisGlobe, GlobeData, data, position, scroll ) {

	// we could try to use thisGlobe to rotate the globe to the clicked coordinates
	// and we could use position to target specific container, if multiple displayContent were selected

	var contentContainer = document.getElementById('itt_globe_content_' + position + '_' + GlobeData.id );

	if ( contentContainer === undefined || contentContainer === null ) {
		return;
	}
	var thisContent = data.content ? data.content : '';

	contentContainer.innerHTML = thisContent;

	if (scroll) {
		var originalTop = Math.floor(contentContainer.getBoundingClientRect().top - 100);
		window.scrollBy({
			top: originalTop,
			left: 0,
			behavior: 'smooth',
		});
	}

}

function contentBelowScroll( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'below', true );
}

function contentBelow( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'below', false );
}

function contentAbove( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'above', false );
}

function contentRight( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'right', false );
}

function contentLeft( thisGlobe, GlobeData, data ) {
	displayContent( thisGlobe, GlobeData, data, 'left', false );
}

function contentLighbox( thisGlobe, GlobeData, data ) {
	openLightbox( thisGlobe, GlobeData, data, 'inline' );
}

function iframeLightbox( thisGlobe, GlobeData, data ) {
	openLightbox( thisGlobe, GlobeData, data, 'iframe' );
}

function openLightbox( thisGlobe, GlobeData, data, type ) {
	//function (id, data, type) {
	var elements = [],
		width = window.innerWidth < 900 ? '90%' : '60%',
		opts = {}; 

	if (type === 'inline') {

		elements.push({
			//href: data.content,
			content: data.content,
			type: type,
			width: width,
			height: 'auto',
		});
	} else if (type === 'iframe') {
		
		elements.push({
			href: data.content,
			type: 'external',
			width: width,
			height: parseInt(window.innerHeight * 0.8),
		});
	}

	opts = {
		touchNavigation: false,
		draggable: false,
		keyboardNavigation: false,
		loopAtEnd: false,
		loop: false,
		zoomable: false,
		elements: elements,
		closeButton: true, 
		closeOnOutsideClick: true,
	};

	// fix for lightbox closing on bigger touch devices
	/*
	if (window.innerWidth > 768 && functionToCheckifTouchScreenDevice()) {
		opts.closeOnOutsideClick = false;
	}
	*/

	// let's make it global for better debugging, just in case
	ittGlobes.lightbox = GLightbox(opts);

	if (data.content !== '' && ittGlobes.lightbox) {
		ittGlobes.lightbox.open();
		ittGlobes.lightboxIsRunning = true;
	} else {
		console.log('Empty Action Content or Incorrect Request - Lightbox not triggered');
	}

	ittGlobes.lightbox.on('close', function () {
		console.log('lightbox closed, we might need to reset stuff');
		ittGlobes.lightboxIsRunning = false;
	});

	// add custom close button to solve issues on touch devices
	ittGlobes.lightbox.on('open', function () {
		console.log('Open event not working?');
		/*
		let close = document.querySelector('.ginner-container .gslide-media .ittglobe_close');
		if (!close) {
		close = document.createElement('span');
		close.classList.add('ittglobe_close');
		close.innerHTML = 'XXX╳';
		let containers = document.querySelectorAll('.ginner-container .gslide-media');
		containers.forEach(function (el) {
			let clone = close.cloneNode(true);
			clone.addEventListener('click', function () {
			lightbox.close();
			});

			el.prepend(clone);
		});
		}
		*/
	});
}

ittGlobes$1.clickActions.DisplayContentBelow = contentBelow;
ittGlobes$1.clickActions.DisplayContentBelowScroll = contentBelowScroll;
ittGlobes$1.clickActions.DisplayContentAbove = contentAbove;
ittGlobes$1.clickActions.DisplayContentRight = contentRight;
ittGlobes$1.clickActions.DisplayContentLeft = contentLeft;
ittGlobes$1.clickActions.DisplayContentLightbox = contentLighbox;
ittGlobes$1.clickActions.DisplayIframeLightbox = iframeLightbox;

/** initialize globe */
ittGlobes$1.pro = function () {

	// check for globe container created by the shortcode
	let globe_render_container = document.getElementsByClassName( 'js-itt-globe-render' );
	for (const globe_container of globe_render_container) {
		if ( ! globe_container.dataset.hasOwnProperty('globe_id') ) {
			continue;
		}
		let globe_id = 0;
		try {
			globe_id = parseInt( JSON.parse(globe_container.dataset.globe_id ) );
		} catch (e) {
			continue;
		}

		if ( ! globe_container.dataset.hasOwnProperty( 'globe_meta' ) ) {
			continue;
		}

		let globe_meta = {};
		try {
			globe_meta = JSON.parse( globe_container.dataset.globe_meta );
		} catch (e) {
			globe_meta = {};
		}

		extend_globe( globe_id, globe_meta );
	}
};

var extend_globe = function ( id, data ) {
	//data types - normal
	let dataTypes = {
		'arcLines': 'arcLineDefaults',
		'regions': 'regionDefaults',
	};

	globe_ut_prepareProData( data );
	globe_ut_setupData( data, dataTypes );

	let thisGlobe = ittGlobes$1.globesIndex[ id ];

	if ( data.globeImage === 'earth-hollow' ) {
		thisGlobe.showGlobe(false).showAtmosphere(false);
	}

	if( data.globeImage === 'noImage' ){
		// return compatible color object
		let globeColor = typeof data.globeColor !== 'undefined' ? data.globeColor : '#064273';
		thisGlobe.globeMaterial().color.set( globeColor );
	}

	if ( data.globeImage === 'customImage' ) {
		let globeCustomImage = typeof data.customImage !== 'undefined' ?  data.customImage.url : false;
		thisGlobe.globeImageUrl( globeCustomImage );
	}

	addAutoRotate( thisGlobe, data );
	addArcsLayer( thisGlobe, data );

	// if there's a live filter, add that info to the main globe
	if( data.layers && Array.isArray(data.layers) && data.liveFilter && data.liveFilter.enabled === "1" ){
		thisGlobe.liveFilter = data.liveFilter;
	}
	
	// overlay layers for Regions, async
	// Wait for the addRegionLayer async operation to complete
	(async () => {
		await Promise.resolve(addRegionLayer(thisGlobe, data));

		// Now that addRegionLayer is complete, we can proceed with the overlay layers
		if (data.layers && Array.isArray(data.layers)) {
			let altitudeMultiplier = 1;
			for (let layer of data.layers) {
				if (layer.regions && Array.isArray(layer.regions)) {
		
					globe_ut_setupData( layer, {'regions': 'regionDefaults'} );

					// fetch regions
					const existingRegions = thisGlobe.originalPolygonsData || [];
					const preparedLayerRegions = await prepareRegions(thisGlobe, layer, altitudeMultiplier);
					const mergedRegions = [...existingRegions, ...preparedLayerRegions];

					altitudeMultiplier++;

					thisGlobe.originalPolygonsData = mergedRegions;
				
					if (data.liveFilter && data.liveFilter.enabled === "1" && parseInt(data.liveFilter.default) !== parseInt(id)) {
						if (parseInt(data.liveFilter.default) === parseInt(layer.id)) {
							// we need to keep base, so merge base also
							if (data.liveFilter.keepBase === "1") {
								let baseAndLayerRegions = [...thisGlobe.basePolygonsData, ...preparedLayerRegions];
								thisGlobe.polygonsData(baseAndLayerRegions);
							} else {
								thisGlobe.polygonsData(layer.regions);
							}
						}
					} else {
						thisGlobe.polygonsData(mergedRegions);
					}
				}
			}
		}
	})();

	// overlay layers for dotLabels, points and arcs
	if( data.layers && Array.isArray( data.layers ) ){

		// Initialize arrays for data collection
		thisGlobe.originalPointsData = thisGlobe.pointsData();
		thisGlobe.originalLabelsData = thisGlobe.labelsData();
		thisGlobe.originalArcsData = thisGlobe.arcsData();

		// store original base map data, if needed
		thisGlobe.basePointsData = thisGlobe.pointsData();
		thisGlobe.baseLabelsData = thisGlobe.labelsData();
		thisGlobe.baseArcsData = thisGlobe.arcsData();

		for( let layer of data.layers ){

			let layerDataTypes = {
				'arcLines': 'arcLineDefaults',
				'points': 'pointDefaults',
				'dotLabels': 'labelDefaults',
			};

			globe_ut_setupData( layer, layerDataTypes );

			// points
			if (layer.points && Array.isArray(layer.points)) {
				const existingPoints = thisGlobe.originalPointsData;
				const mergedPoints = [...existingPoints, ...layer.points];

				thisGlobe.originalPointsData = mergedPoints;
				
				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.pointsData(layer.points);
						// we need to keep base, so merge base also
						if( data.liveFilter.keepBase === "1" ){
							let baseAndLayerPoints = [...thisGlobe.basePointsData, ...layer.points];
							thisGlobe.pointsData( baseAndLayerPoints );
						}
					}
				} else {
					thisGlobe.pointsData(mergedPoints);
				}
			}

			// dot labels
			if (layer.dotLabels && Array.isArray(layer.dotLabels)) {
				const existingLabels = thisGlobe.originalLabelsData;
				const mergedLabels = [...existingLabels, ...layer.dotLabels];

				thisGlobe.originalLabelsData = mergedLabels;
				
				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.labelsData(layer.dotLabels);
						// we need to keep base, so merge base also
						if( data.liveFilter.keepBase === "1" ){
							let baseAndLayerDotLabels = [...thisGlobe.baseLabelsData, ...layer.dotLabels];
							thisGlobe.labelsData( baseAndLayerDotLabels );
						}
					}
				} else {
					thisGlobe.labelsData(mergedLabels);
				}
			}

			// arcs
			if (layer.arcLines && Array.isArray(layer.arcLines)) {
				const existingArcs = thisGlobe.originalArcsData;
				const mergedArcs = [...existingArcs, ...layer.arcLines];

				thisGlobe.originalArcsData = mergedArcs;
				
				if( data.liveFilter && data.liveFilter.enabled === "1" && parseInt( data.liveFilter.default ) !== parseInt( id ) ){
					if( parseInt( data.liveFilter.default ) === parseInt( layer.id ) ){
						thisGlobe.arcsData(layer.arcLines);
						// we need to keep base, so merge base also
						if( data.liveFilter.keepBase === "1" ){
							let baseAndLayerArcs = [...thisGlobe.baseArcsData, ...layer.arcLines];
							thisGlobe.arcsData( baseAndLayerArcs );
						}
					}
				} else {
					thisGlobe.arcsData(mergedArcs);
				}
			}
			
			// regions are done async
		}
	}

	// if admin
	if ( ittGlobeData.isAdmin ) {

		thisGlobe.onGlobeReady(function () {
			const event = new Event("adminProGlobeReady");
			document.dispatchEvent(event);
		});
	}

	// to make it publically accessible
	thisGlobe.meta = data;
};

ittGlobes$1.pro();
