/**
 * Prepares safe url
 * @param string str url to prepare
 * @returns string prepared URL
 */
const globe_ut_prepareURL = function (str) {

	if (typeof str !== "string") {
		return str;
	}

	// parse html entities
	str = str.replace(/&amp;/gi, '&');
	str.replace(/&#(\d+);/g, function (match, dec) {
		return String.fromCharCode(dec);
	});

	// check if allowed url
	var url, protocols;
	try {
		url = new URL(str);
	} catch (_) {
		url = false;
	}

	// check if valid url. If string it might be a anchor or relative path, so keep going
	if (url) {
		// acepted protocols
		protocols = [null, "http:", "https:", "mailto:", "tel:"];
		if (!protocols.includes(url.protocol)) {
			console.log('URL protocol not allowed');
			return '';
		}
	}

	return str;

};

/** Setup click events for element */
function setupClickEvent( thisGlobe, GlobeData, element ) {

	let layerDataTypes = {
		'arcLines': 'arcLineDefaults',
		'points': 'pointDefaults',
		'dotLabels': 'labelDefaults',
		'regions': 'regionDefaults',
	};

	if(element.type === 'regions'){
		element = { ...element.properties };
	}

	if (new URLSearchParams(window.location.search).has('debug')) {
		console.debug(element);
	}

	// if we arrive here and it's still set as default, we need to set it to the default action for that layer
	if( element.action === 'default' ){
		element.action = GlobeData[layerDataTypes[element.type]].action;
	}

	if (element.action === "none") {
		// do nothing
		return;
	}

	const clickAction = element.action;
	if (clickAction && typeof window.ittGlobes.clickActions[clickAction] === 'function') {
		window.ittGlobes.clickActions[clickAction]( thisGlobe, GlobeData, element );
	}
	if ( ittGlobeData.isAdmin ) {
		return;
	}

	// check urls
	if (element.action === "open_url" || element.action === "open_url_new") {
		element.content = globe_ut_prepareURL(element.content);
	}
	// open new url
	if (element.action === "open_url" && element.content !== "") {
		document.location = element.content;
	} else if (element.action === "open_url_new" && element.content !== "") {
		window.open(element.content);
	}
}

let ittGlobesAdminPro = {};

ittGlobesAdminPro.init = function(){
	ittGlobesAdminPro.setupGeojson();
	ittGlobesAdminPro.updateGeojson();
	ittGlobesAdminPro.setupGlobeReady();
	ittGlobesAdminPro.sortableLayer();
};

ittGlobesAdminPro.setupGeojson = function(){

	let geojsonSelect = document.getElementsByName( 'globe_info[regionSource]' );
	if ( typeof geojsonSelect === 'undefined' || geojsonSelect.length === 0 ) {
		return;
	}

	geojsonSelect = geojsonSelect[0];
	if( typeof geojsonSelect === 'undefined'){
		return;
	}

	geojsonSelect.addEventListener( 'change', function( ev ){
		ittGlobesAdminPro.updateGeojson( ev.target.value );
	});

	// when cpt list updates
	window.addEventListener( 'message', function( event ) {

		if ( ! event.data.hasOwnProperty( 'message' ) ||
			! event.data.hasOwnProperty( 'params' ) ) {
				return;
		}
		if ( event.data.message !== 'refresh-cpt-list' ) {
			return;
		}

		const params = event.data.params;
		if ( ! params.hasOwnProperty( 'post_type' ) &&
			! params.hasOwnProperty( 'parent_id' ) ) {
				return;
		}

		setTimeout( function(){
			ittGlobesAdminPro.updateGeojson();
		}, 500 );
	});
};


ittGlobesAdminPro.getGeojsonUrlVal = function(){
	let geojsonUrlSelect = document.getElementsByName( 'globe_info[regionCustomSourceURL]' );
	if ( typeof geojsonUrlSelect === 'undefined' || geojsonUrlSelect.length == 0 ) {
		return;
	}

	geojsonUrlSelect = geojsonUrlSelect[0];
	if ( typeof geojsonUrlSelect === 'undefined' ) {
		return;
	}

	return geojsonUrlSelect.value;
};

ittGlobesAdminPro.getGeojsonVal = function(){
	let geojsonSelect = document.getElementsByName( 'globe_info[regionSource]' );
	if ( typeof geojsonSelect === 'undefined' || geojsonSelect.length == 0 ) {
		return;
	}

	geojsonSelect = geojsonSelect[0];
	if ( typeof geojsonSelect === 'undefined' ) {
		return;
	}

	return geojsonSelect.value;
};

ittGlobesAdminPro.updateGeojson = function( val ){
	if ( typeof val === 'undefined' ) {
		val = ittGlobesAdminPro.getGeojsonVal();
	}


	let urlVal = '';

	if( val === 'custom' ){
		urlVal = ittGlobesAdminPro.getGeojsonUrlVal();
	}

	let addNewButton = document.querySelector ('#block_itt_globe_region a.js-modal-add-new-cpt' );
	addNewButton.dataset.geojson = val;
	addNewButton.dataset.geojsonURL = urlVal;

	// update cpt table
	let buttons = document.querySelectorAll( '#block_itt_globe_region .js-modal-edit-cpt' );
	buttons.forEach(function( button ){
		button.dataset.geojson = val;
	} );

	// update 'add new' preview button
	let addNewSelectButton = ittGlobesAdmin.infoContainer.querySelector('#itt_globe_preview_data_clicked .itt_globe_preview_action_add_new');
	addNewSelectButton.dataset.geojson = val;

};


/**
 * Runs when globe is ready to listen for changes and update info
 */
ittGlobesAdminPro.setupGlobeReady = function () {
	// trigger when globe is ready
	document.addEventListener("adminProGlobeReady", function () {

		// first and unique globe stored
		let thisGlobePro = Object.values( ittGlobes.globesIndex )[0];

		// Add lines option to add new entry select
		let addNewButton = ittGlobesAdmin.infoContainer.querySelector('#itt_globe_preview_data_clicked .itt_globe_preview_action_add_new');

		if( ! ittGlobesAdminPro.hasOption( addNewButton, 'itt_globe_line' ) ){
			let newOption = new Option('Line', 'itt_globe_line' );
			addNewButton.add(newOption,undefined);
		}

		thisGlobePro.onPolygonClick(function (region, event, coordinates) {

			coordinates.latitude = ittGlobesAdmin.round( coordinates.lat );
			coordinates.longitude = ittGlobesAdmin.round( coordinates.lng );

			ittGlobesAdmin.preview.clickedCoordinates = coordinates;

			region.properties.coordinates = coordinates;

			// ripple click
			const srcRing = { lat: coordinates.latitude, lng: coordinates.longitude };
			thisGlobePro.ringsData([...thisGlobePro.ringsData(), srcRing]);
			setTimeout(() => thisGlobePro.ringsData(thisGlobePro.ringsData().filter(r => r !== srcRing)), 1000 * 0.4);

			// if clicked empty region, set title
			if( typeof region.properties.title === 'undefined' ){
				region.properties.title = region.properties.name || region.id;

				// also set add new extra entry to add regions
				let addNewButton = ittGlobesAdmin.infoContainer.querySelector('#itt_globe_preview_data_clicked .itt_globe_preview_action_add_new');

				// reset dataset
				Object.keys(addNewButton.dataset).forEach(dataKey => {
					delete addNewButton.dataset[dataKey];
				});

				if( ! ittGlobesAdminPro.hasOption( addNewButton, 'itt_globe_region' ) ){
					let newOption = new Option('Region', 'itt_globe_region' );
					addNewButton.add(newOption,undefined);
				}
			}

			// set dataset values to carry on to add new url
			ittGlobesAdmin.preview.clickedDataset = {
				region_id : region.properties.id || region.id,
				title : region.properties.name,
				geojson : ittGlobesAdminPro.getGeojsonVal(),
				geojsonURL : ittGlobesAdminPro.getGeojsonUrlVal()
			};

			ittGlobesAdmin.preview.clickedElement = Object.assign({}, region.properties);
			ittGlobesAdmin.preview.clickedElement.type = 'Location in Region';
			ittGlobesAdmin.preview.clickedElement.post_type = 'itt_globe_region';
			delete ittGlobesAdmin.preview.clickedElement.__threeObj;
			ittGlobesAdmin.updateClickedPreviewInfo();

			const globe_meta = JSON.parse( ittGlobesAdmin.meta );
			setupClickEvent( thisGlobePro, globe_meta, region.properties );
		});
	});
};

// Function to check if a specific select element already has an option with a specific value
ittGlobesAdminPro.hasOption = function(selectElement, valueToCheck) {
    // Get all options of the select element
    const options = selectElement.options;

    // Loop through all options
    for (let i = 0; i < options.length; i++) {
        // Check if the value of the current option matches the value we are checking for
        if (options[i].value === valueToCheck) {
            // If a match is found, return true
            return true;
        }
    }

    // If no match is found, return false
    return false;
};


ittGlobesAdminPro.sortableLayer = function () {

	// Using jQuery since we'll use jquery-ui for sortable anyway
	var layerParent = jQuery('[data-depend-id=layer]');
	var layerSortable;

	if (layerParent) {
		layerSortable = layerParent.first().closest('ul');

		// take advantage of this function to also set the title attribute for the li with the map id
		layerSortable.find('li').each(function () {
			jQuery(this).attr('title', 'Globe ID: ' + jQuery(this).find('input').attr('value'));
		});

	}

	// update order when we select a new layer
	layerParent.on('change', function () {
		ittGlobesAdminPro.updateOrderlayer();
	});

	// if there's an order
	var layerOrder = document.querySelector('[data-depend-id=layerOrder]');

	if (layerOrder && layerOrder.value !== '' && layerSortable) {
		
		var orderValues = JSON.parse(layerOrder.value);
		if (Array.isArray(orderValues)) {
			orderValues.reverse();
			orderValues.forEach(function (val) {
				var thisOrder = layerSortable.find("input[value=" + val + "]").closest('li');
				if (thisOrder) {
					layerSortable.prepend(thisOrder);
				}
			});
		}
	}

	// set sortable behaviour
	if (layerParent) {
		layerSortable.sortable({
			placeholder: "ui-state-highlight",
			stop: function (event, ui) {
				ittGlobesAdminPro.updateOrderlayer();
				
				// function when layer is sorted
			}
		});
		layerParent.disableSelection();
	}
};

ittGlobesAdminPro.updateOrderlayer = function () {
	var layerOrder = document.querySelector('[data-depend-id=layerOrder]');
	var layerSelected = document.querySelectorAll('input[data-depend-id=layer]:checked');
	var order = [];

	if (layerSelected) {
		layerSelected.forEach(function (el) {
			order.push(el.value);
		});
	}
	order = JSON.stringify(order);
	layerOrder.value = order;

};

ittGlobesAdminPro.init();
