<?php
namespace Saltus\WP\Plugin\InteractiveGlobes\Features\SearchDropdown;

if ( defined( 'SALTUS_PLAN' ) && SALTUS_PLAN !== 'pro' ) {
	return [];
}

use Saltus\WP\Plugin\InteractiveGlobes\Saltus\WP\Framework\Infrastructure\Plugin\Registerable;
use Saltus\WP\Plugin\InteractiveGlobes\Services\Assets\AssetsCore;
use Saltus\WP\Plugin\InteractiveGlobes\Project;
use Saltus\WP\Plugin\InteractiveGlobes\Saltus\WP\Framework\Infrastructure\Container\Invalid;
use Saltus\WP\Plugin\InteractiveGlobes\Saltus\WP\Framework\Infrastructure\Container\ServiceContainer;

/**
 * The Meta class
 */
class SearchDropdownCore implements Registerable {

	private $services;
	private $project;
	private $meta;
	private $options;

	public function __construct( ...$dependencies ) {

		if ( empty( $dependencies[0] ) ) {
			throw Invalid::from( 'Project' );
		}
		if ( ! $dependencies[0] instanceof Project ) {
			throw Invalid::from( $dependencies[0] );
		}

		if ( empty( $dependencies[1] ) ) {
			throw Invalid::from( 'ShortcodeCore' );
		}
		if ( ! $dependencies[1] instanceof ServiceContainer ) {
			throw Invalid::from( $dependencies[1] );
		}

		$this->project  = $dependencies[0];
		$this->services = $dependencies[1];
	}

	/**
	 * Register SearchDropdown filters
	 */
	public function register() {
		// globe search dropdown
		add_action( 'itt_globes/render/post_setup_meta', [ $this, 'check_search_dropdown' ], 60, 2 );

		$this->options = get_option( 'ittglobes' );
	}

	public function check_search_dropdown( $meta, $id ) {

		if ( $id === 0 ) {
			return $meta;
		}
		if ( empty( $meta['search'] ) || boolval( $meta['search']['enabled'] ) === false ) {
			return $meta;
		}

		$hook = 'itt_globes/render/content_before';
		if ( $meta['search']['position'] === 'after' ) {
			$hook = 'itt_globes/render/content_after';
		}

		$meta['id'] = $id;
		$this->meta = $meta;

		add_action( $hook, [ $this, 'search_content' ], 10, 2 );
		$this->register_assets();

		return $meta;
	}

	public function search_content( $content, $id ) {

		$mobile_only_class = '';
		if ( isset( $this->meta['search']['mobile_only'] ) && boolval( $this->meta['search']['mobile_only'] ) === true ) {
			$mobile_only_class = ' itt_globe_search_dropdown_mobile_only';
		}

		if ( isset( $this->meta['search']['choices'] ) && boolval( $this->meta['search']['choices'] ) === true ) {
			$class = 'itt_globe_search_dropdown_choices';
		} else {
			$class = 'itt_globe_search_dropdown';
		}

		$properties         = false;
		$include_properties = isset( $this->meta['search']['properties'] ) && ! empty( $this->meta['search']['properties'] ) ? true : false;

		if ( $include_properties ) {
			$properties = explode( ',', $this->meta['search']['properties'] );
		}

		$nomatches         = $this->meta['search']['no_matches_text'];
		$select            = $this->meta['search']['select_text'];
		$placeholder       = $this->meta['search']['placeholder_text'];
		$custom_properties = $this->meta['search']['properties'];

		$search_html = sprintf(
			'<select class="%s %s" globe-id="%s" data-select="%s" data-nomatches="%s" data-placeholder="%s" id="itt_globe_search_dropdown_%s" data-custom-properties="%s">',
			esc_attr( $class ),
			esc_attr( $mobile_only_class ),
			esc_attr( $id ),
			esc_attr( $select ),
			esc_attr( $nomatches ),
			esc_attr( $placeholder ),
			esc_attr( $id ),
			esc_attr( $custom_properties )
		);

		$search_html .= '<option value="">' . esc_html( $this->meta['search']['placeholder_text'] ) . '</option>';

		$search_entries = [];
		$label_property = ! empty( $this->meta['search']['label_property'] ) ? $this->meta['search']['label_property'] : 'title';

		// Add points
		if ( ! empty( $this->meta['points'] ) ) {
			foreach ( $this->meta['points'] as $point ) {
				$label      = isset( $point[ $label_property ] ) ? $point[ $label_property ] : $point['id'];
				$this_props = [];
				if ( $include_properties && ! empty( $properties ) ) {
					foreach ( $properties as $property ) {
						$property = trim( $property );
						if ( isset( $point[ $property ] ) ) {
							$this_props[ $property ] = $point[ $property ];
						}
					}
				}
				$search_entries[] = [
					'id'         => $point['id'],
					'label'      => $label,
					'properties' => $this_props,
				];
			}
		}

		// Add dot labels
		if ( ! empty( $this->meta['dotLabels'] ) ) {
			foreach ( $this->meta['dotLabels'] as $dotlabel ) {
				$label = isset( $dotlabel[ $label_property ] ) ? $dotlabel[ $label_property ] : $dotlabel['id'];

				$this_props = [];
				if ( $include_properties && ! empty( $properties ) ) {
					foreach ( $properties as $property ) {
						$property = trim( $property );
						if ( isset( $dotlabel[ $property ] ) ) {
							$this_props[ $property ] = $dotlabel[ $property ];
						}
					}
				}

				$search_entries[] = [
					'id'         => $dotlabel['id'],
					'label'      => $label,
					'properties' => $this_props,
				];
			}
		}

		// Add regions
		if ( ! empty( $this->meta['regions'] ) ) {
			$label_property = ! empty( $this->meta['search']['label_property'] ) ? $this->meta['search']['label_property'] : 'title';
			foreach ( $this->meta['regions'] as $region ) {
				$label      = isset( $region[ $label_property ] ) ? $region[ $label_property ] : $region['id'];
				$this_props = [];
				if ( $include_properties && ! empty( $properties ) ) {
					foreach ( $properties as $property ) {
						$property = trim( $property );
						if ( isset( $region[ $property ] ) ) {
							$this_props[ $property ] = $region[ $property ];
						}
					}
				}

				// for regions, the 'id' is an array, so we need to convert it to a string
				$region_id        = implode( ',', $region['id'] );
				$search_entries[] = [
					'id'         => $region_id,
					'label'      => $label,
					'properties' => $this_props,
				];
			}
		}

		// Create options from the array
		foreach ( $search_entries as $entry ) {
			$search_html .= '<option value="' . $entry['id'] . '" data-custom-properties="' . esc_attr( wp_json_encode( $entry['properties'] ) ) . '">' . esc_html( $entry['label'] ) . '</option>';
		}

		$search_html .= '</select>';

		$search_html = '<div class="itt_globe_search_dropdown_wrapper">' . $search_html . '</div>';

		return $search_html . $content;
	}

	public function register_assets() {
		try {
			$assets = $this->services->get( 'assets' )->create();
		} catch ( \Exception $exception ) {

			if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions
				error_log( 'Failed to load styles' );
			}
		}

		$name = $assets->register_script(
			'/assets/js/vendor/public/choices.js',
			[],
			true
		);
		wp_enqueue_script( $name );

		$name = $assets->register_script(
			'/assets/js/public/features/search/search.js',
			[ '/assets/js/vendor/public/choices.js' ],
			true
		);
		wp_enqueue_script( $name );

		$name = $assets->register_style(
			'/assets/css/features/search/search.css',
			[]
		);
		wp_enqueue_style( $name );

		$name = $assets->register_style(
			'/assets/css/vendor/public/choices.css',
			[]
		);
		wp_enqueue_style( $name );
	}
}
