<?php
namespace Saltus\WP\Plugin\InteractiveGlobes\Features\RegionCounter;

if ( defined( 'SALTUS_PLAN' ) && SALTUS_PLAN !== 'pro' ) {
	return [];
}

use Saltus\WP\Plugin\InteractiveGlobes\Saltus\WP\Framework\Infrastructure\Plugin\Registerable;
use Saltus\WP\Plugin\InteractiveGlobes\Saltus\WP\Framework\Infrastructure\Container\Invalid;
use Saltus\WP\Plugin\InteractiveGlobes\Saltus\WP\Framework\Infrastructure\Container\ServiceContainer;
use Saltus\WP\Plugin\InteractiveGlobes\Project;
use Saltus\WP\Plugin\InteractiveGlobes\Services\Model\Base\Settings;
/**
 * The Region Counter class
 */
class RegionCounterCore implements Registerable {

	private $services;
	private $project;
	private $meta;
	private $options;

	public function __construct( ...$dependencies ) {

		if ( empty( $dependencies[0] ) ) {
			throw Invalid::from( 'Project' );
		}
		if ( ! $dependencies[0] instanceof Project ) {
			throw Invalid::from( $dependencies[0] );
		}

		if ( empty( $dependencies[1] ) ) {
			throw Invalid::from( 'ShortcodeCore' );
		}
		if ( ! $dependencies[1] instanceof ServiceContainer ) {
			throw Invalid::from( $dependencies[1] );
		}

		$this->project  = $dependencies[0];
		$this->services = $dependencies[1];
	}

	/**
	 * Register Region Counter
	 */
	public function register() {
		// add the region counter
		add_action( 'itt_globes/render/post_setup_meta', [ $this, 'add_region_counter' ], 50, 2 );

		$this->options = get_option( 'ittglobes' );
	}

	public function add_region_counter( $meta, $id ) {

		$this->meta = $meta;

		if ( empty( $meta['regionCount'] ) || boolval( $meta['regionCount']['enable'] ) === false ) {
			return $meta;
		}       $template = $meta['regionCount']['template'];
		$position         = $meta['regionCount']['position'];

		$hook = 'itt_globes/render/content_before';

		if ( $position === 'bottom' ) {
			$hook = 'itt_globes/render/content_after';
		}

		add_filter( 'itt_globes/render/container_class', [ $this, 'container_class' ] );

		add_action( $hook, [ $this, 'render_region_counter' ], 10, 2 );

		return $meta;
	}

	public function render_region_counter( $content, $id ) {

		$meta          = $this->meta;
		$template      = $meta['regionCount']['template'];
		$counter_html  = '';
		$geojson       = isset( $meta['regionSource'] ) ? $meta['regionSource'] : 'continents.json';
		$count         = count( $meta['regions'] );
		$total_regions = 0;
		$path          = dirname( dirname( __DIR__ ) );
		$settings      = Settings::get( $path );
		$assets_url    = $settings['assets_url'];

		if ( $geojson !== 'custom' ) {
			$geojson_url = $assets_url . 'assets/geojson/' . $geojson;
			if ( ! preg_match( '/(\.json|\.geojson)$/i', $geojson ) ) {
				$geojson_url .= '.json';
			}
		} else {
			$geojson_url = isset( $_GET['geojsonURL'] ) ? esc_attr( $_GET['geojsonURL'] ) : false;
		}

		$request = wp_remote_get( $geojson_url );
		// Decode the JSON file
		if ( ! is_wp_error( $request ) ) {
			$json      = wp_remote_retrieve_body( $request );
			$json_data = json_decode( $json, true );

			if ( ! empty( $json_data['features'] ) ) {

				// Update count to only include regions that exist in both arrays
				$count         = 0;
				$total_regions = count( $json_data['features'] );

				// Create a lookup array for the region IDs and names, making it easier to compare
				$feature_lookup = [];
				foreach ( $json_data['features'] as $feature ) {
					$feature_lookup[ $feature['id'] ]                 = true;
					$feature_lookup[ $feature['properties']['name'] ] = true;
				}

				foreach ( $meta['regions'] as $region ) {
					$region_id = $region['id'];
					// Handle array or comma-separated string IDs
					if ( is_array( $region_id ) ) {
						$region_ids = $region_id;
					} else {
						$region_ids = array_map( 'trim', explode( ',', $region_id ) );
					}

					foreach ( $region_ids as $id ) {
						if ( isset( $feature_lookup[ $id ] ) ) {
							++$count;
							break; // Move to the next region after a match is found
						}
					}
				}
			}
		}

		// maybe this is a possibility. If the total regions is 0, then it can use the count of the regions
		if ( $total_regions === 0 ) {
			$total_regions = $count;
		}

		// if still zero, means nothing was available and percentage should be 0
		$percentage = 0;
		if ( $total_regions !== 0 ) {
			$percentage = round( $count / $total_regions * 100, 2 );
		}

		$counter_html = str_replace(
			[ '{regionCount}', '{totalRegions}', '{regionPercentage}' ],
			[
				'<span class="itt-globe-region-count">' . $count . '</span>',
				'<span class="itt-globe-region-total">' . $total_regions . '</span>',
				'<span class="itt-globe-region-percentage">' . $percentage . '</span>',
			],
			$template
		);

		$counter_html = '<div class="itt-globe-region-counter">' . $counter_html . '</div>';

		return $counter_html . $content;
	}

	public function container_class( $container_class ) {
		return $container_class . ' itt_globe_wrapper_with_region_counter';
	}
}
